#ifndef AHLGREN_TOKENIZER
#define AHLGREN_TOKENIZER

#include <iostream>
#include <list>
#include <stack>
#include <stdexcept>

#include "useful.h"
#include "global.h"
#include "trie.h"


namespace lp {
    using namespace std;
    using namespace useful;

    class Functor; // class declaration
	class Dictionary;

    class parse_error {
        string err;
    public:
        parse_error(const string& s) : err(s) {}
        const string& what() const { return err; }
    };

    
    class Token {
    public:
        typedef Functor node_type;
        typedef unsigned token_type;
		// Null
        static const token_type Null = 0x0;
		// Constant type (not numeric types)
        static const token_type Constant = 0x1;
        // operator types
        static const token_type fx = 0x2;
        static const token_type fy = 0x4;
        static const token_type yfx = 0x8;
        static const token_type xfy = 0x10;
        static const token_type xfx = 0x20;
        static const token_type xf = 0x40;
        static const token_type yf = 0x80;
        // others
        static const token_type Leftp = 0x100;
        static const token_type Rightp = 0x200;
        static const token_type Variable = 0x400;
        static const token_type Function = 0x800;
        static const token_type LeftList = 0x1000;
        static const token_type RightList = 0x2000;
        static const token_type ListSep = 0x4000;
        static const token_type SeqSep = 0x8000;
        // numeric
        static const token_type Integer = 0x20000;
        static const token_type Double = 0x40000;
		// new line
        static const token_type Newline = 0x80000;

		// Default constructor
		Token() : _id(0), _type(Null), _prec(0) {}
		// Construct operator, defaults to prefix function with unknown arity
        Token(id_type i, token_type t, int prec = 0) : _id(i), _type(t), _prec(prec) {}
		// Construct a token from a name
        Token(const string& s, token_type t, int prec = 0);
 		// Construct a token from an integer (use dummy bool to distinguish from id_type)
		static Token make_int(long long v);
        static Token make_double(double v);

        // Compare two operators
        bool operator==(const Token& t) const {
            return id() == t.id() && type() == t.type() && precedence() == t.precedence();
        }
        bool operator!=(const Token& t) const { return !(*this == t); }

		// Reset to new token
		void reset(id_type i, token_type t, int p) { _id = i ; _type = t; _prec = p; }

		// Get id
		id_type id() const { return _id; }

		// Get data
		data get_data() const;

        // Get symbol (assumes it is a string)
        string symbol() const;

		// Get/Set precedence
        int precedence() const { return _prec; }
        void set_precedence(int i) { _prec = i; }

		// Get/set type (fy, yfx, etc)
        token_type type() const { return _type; }
        bool type(token_type t) const { return (_type & t) != 0; }
        void set_type(token_type t) { _type = t; }

		// Simpler versions of type()
        bool is_variable() const { return _type == Variable; }
		bool is_constant() const { return type(Constant | Integer | Double); }
		bool is_integer() const { return _type == Integer; }
		bool is_double() const { return _type == Double; }
		bool is_numeric() const { return type(Integer | Double); }
        bool is_function() const { return _type == Function; }
        //bool is_functor() const { return is_operator() || is_function() || is_constant() || is_variable(); }
        bool is_operator() const { return type(fx | fy | yfx | xfy | xfx | xf | yf); }
        bool is_prefix_op() const { return type(fx | fy); }
        bool is_suffix_op() const { return type(xf | yf); }
        bool is_binary_op() const { return type(yfx | xfy | xfx); }
        bool has_arg_left() const { return type(xf | yf | xfx | xfy | yfx); }
        bool has_arg_right() const { return type(fx | fy | xfx | xfy | yfx); }
        bool is_yfx() const { return _type == yfx; }
        bool is_xfy() const { return _type == xfy; }

        bool is_xfx() const { return _type == xfx; }
        bool is_fy() const { return _type == fy; }
        bool is_fx() const { return _type == fx; }
        bool is_yf() const { return _type == yf; }
        bool is_xf() const { return _type == xf; }
        bool is_lp() const { return _type == Leftp; }
        bool is_rp() const { return _type == Rightp; }
        bool is_ll() const { return _type == LeftList; }
        bool is_rl() const { return _type == RightList; }
        bool is_list_sep() const { return _type == ListSep; }
        bool is_seq_sep() const { return _id == sequence_id; } // special operator
        bool is_null() const { return _type == Null; }
		bool is_newline() const { return _type == Newline; }

		// Helpers
		// Left and Right delimiter tokens
		bool is_ldel() const { return is_seq_sep() || is_lp() || is_ll(); }
		bool is_rdel() const { return is_seq_sep() || is_rp() || is_rl(); }

		// Create copy
        Token* create() const { return new Token(); }
        Token* copy() const { return new Token(*this); }
	protected:
		id_type _id;
        token_type _type; // type of token (see enum definition)
        int _prec; // precedence
    };


    // Exception class for Dictionary
    struct dic_not_found : public std::runtime_error {
        dic_not_found() : runtime_error("Dictionary word not found") {}
    };

    
    class Dictionary {
    public:
        typedef Token::token_type token_type;

        class Prolog_syntax {};

		// Construct Empty Dictionary
        Dictionary() : tokt(new Trie<Token>) {}

		// Destructor
        ~Dictionary() { delete tokt; }
        // Construct Prolog Dictionary
        Dictionary(Prolog_syntax);

		// Move Constructor
        Dictionary(Dictionary&& dic) : tokt(dic.tokt) { dic.tokt = nullptr; }
		// Move assignment
        //Dictionary& operator=(Dictionary&& dic) { delete tokt; tokt = dic.tokt; dic.tokt = nullptr; return this; }

		// Add token
		// Add or Overwrite token
        void add(const Token& t) { 
			Trie<Token>* ptr = tokt->get_entry(t.symbol());
			if (ptr) {
				// Check if we already have entry
				auto rng = ptr->equal_range();
				auto i = rng.first;
				// We store class prefix/infix/suffix, so new Tokens can overwrite previous from same class
				enum class pos { prefix, infix, suffix };
				auto token_class = [](const Token& tok) -> pos {
					if (tok.is_prefix_op()) return pos::prefix;
					else if (tok.is_binary_op()) return pos::infix;
					else { assert(tok.is_suffix_op()); return pos::suffix; }
				};
				pos tpos = token_class(t);
				for ( ; i != rng.second; ++i) {
					assert(i->symbol() == t.symbol());
					if (token_class(*i) == tpos) {
						// Symbol and type are the same class, overwrite precedence
						i->set_type(t.type()); // same type, not just class
						i->set_precedence(t.precedence());
						break;
					}
				}
				if (i == rng.second) {
					ptr->insert_here(t);
				}
			} else {
				// No entry, insert
				// New definition, insert
				tokt->insert(t.symbol(),t); 
			}
		}

		// Has symbol for some token?
        bool has(const string& s) const { return tokt->contains(s) != nullptr; }

		// Get precedence of token
        int precedence(const string& s, token_type t) const {
            auto p = tokt->equal_range(s);
            for (auto i = p.first; i != p.second; ++i) {
                if (i->type(t)) return i->precedence();
            }
            throw dic_not_found();
        }

		// Get any token with symbol s (or throw not_found)
        Token get(const string& s) const {
            const auto p = tokt->equal_range(s);
            if (p.first == p.second) throw dic_not_found();
            return *p.first;
        }

        // Get infix version of symbol
        token_type infix_op(const string& s) const {
            auto p = tokt->equal_range(s);
            for (auto i = p.first; i != p.second; ++i) {
                if (i->is_binary_op()) return i->type();
            }
            throw dic_not_found();
        }

        bool has_infix_op(const string& s) const {
            try {
                infix_op(s);
                return true;
            } catch (dic_not_found) {
                return false;
            }
        }

		// Get prefix version of symbol
        token_type prefix_op(const string& s) const {
            auto p = tokt->equal_range(s);
            for (auto i = p.first; i != p.second; ++i) {
                if (i->is_prefix_op()) return i->type();
            }
            throw dic_not_found();
        }

        bool has_prefix_op(const string& s) const {
            try {
                prefix_op(s);
                return true;
            } catch (dic_not_found) {
                return false;
            }
        }

		// Get suffix version of symbol
        token_type suffix_op(const string& s) const {
            auto p = tokt->equal_range(s);
            for (auto i = p.first; i != p.second; ++i) {
                if (i->is_suffix_op()) return i->type();
            }
            throw dic_not_found();
        }

        bool has_suffix_op(const string& s) const {
            try {
                suffix_op(s);
                return true;
            } catch (dic_not_found) {
                return false;
            }
        }

        const Trie<Token>* get_trie() const {
            return tokt;
        }
    private:
        // Members
        Trie<Token>* tokt;
        // Forbid copy constructor
        Dictionary(const Dictionary&);
        Dictionary& operator=(const Dictionary&);
    };
	extern Dictionary prolog_dic;

	//=================== Declare Global Parsing Functions =======================//
	Functor* stream_to_expr(istream& is, long long& linec, std::set<id_type>& qvars, const Dictionary& dic = prolog_dic);
	inline Functor* stream_to_expr(istream& is, long long& l, const Dictionary& dic = prolog_dic) { std::set<id_type> s; return stream_to_expr(is,l,s,dic); }
	inline Functor* stream_to_expr(istream& is, const Dictionary& dic = prolog_dic) { long long l=1; std::set<id_type> s; return stream_to_expr(is,l,s,dic); }
    std::list<Token> tokenize(istream& is, long long& line, std::set<id_type>& qvars, const Dictionary& dic);
	Token read_atom(istream& is, long long& linec, std::set<id_type>& qvars, const Dictionary& dic);
    void tokens_to_operators(list<Token>& tokenq, long long& linec, const Dictionary& dic);
	bool is_binop_as_function(list<Token>::const_iterator i, 
		list<Token>::const_iterator end, const Dictionary& dic);
    list<Token> infix_to_postfix(list<Token>& tokenq, long long& l, id_type end1 = 0, id_type end2 = 0, id_type end3 = 0);
    Functor* postfix_to_tree(list<Token>& tokenq);
     

	//=================== Inline Member Functions =======================//	
} // namespace lp
#include "functor.h"
namespace lp {
	inline Token Token::make_int(long long v) { Token t; t._id = Functor::index(v); t._type = Integer; t._prec = 0; return t; }
	inline Token Token::make_double(double v) { Token t; t._id = Functor::index(v); t._type = Double; t._prec = 0; return t; }
	inline Token::Token(const string& s, token_type t, int prec) : _id(Functor::index(s)), _type(t), _prec(prec) {}
	inline data Token::get_data() const { return Functor::get_data(id()); }
	inline string Token::symbol() const 
	{
		const data d = Functor::get_data(id());
		switch (d.type()) {
		case 0: return string();
		case data::int_tag: return stringify(d.get_int());
		case data::float_tag: return stringify(d.get_float());
		default: return d.get_atom();
		}
	}

} // namespace lp


#endif

